const axios = require('axios');

const BASE_URL = 'http://localhost:5500/api/interests';

// Test function to check all interests endpoints
async function testInterestsAPI() {
    console.log('🧪 Testing Interests API...\n');

    try {
        // Test 1: Get all interests
        console.log('1. Testing GET /api/interests');
        const allInterests = await axios.get(`${BASE_URL}`);
        console.log(`✅ Success: Found ${allInterests.data.data.length} interests`);
        console.log(`📊 Total Count: ${allInterests.data.count} interests`);
        console.log('');

        // Test 2: Get interests with search
        console.log('2. Testing GET /api/interests?search=programming');
        const searchResults = await axios.get(`${BASE_URL}?search=programming`);
        console.log(`✅ Success: Found ${searchResults.data.data.length} programming-related interests`);
        console.log('');

        // Test 3: Get interests by category
        console.log('3. Testing GET /api/interests/categories');
        const categories = await axios.get(`${BASE_URL}/categories`);
        console.log(`✅ Success: Found ${categories.data.data.length} categories`);
        console.log(`📂 Categories: ${categories.data.data.slice(0, 5).join(', ')}...`);
        console.log('');

        // Test 4: Get interests by specific category
        console.log('4. Testing GET /api/interests/category/Technology');
        const techInterests = await axios.get(`${BASE_URL}/category/Technology`);
        console.log(`✅ Success: Found ${techInterests.data.data.length} technology interests`);
        console.log('');

        // Test 5: Get single interest by ID
        if (allInterests.data.data.length > 0) {
            const firstInterestId = allInterests.data.data[0]._id;
            console.log('5. Testing GET /api/interests/:id');
            const singleInterest = await axios.get(`${BASE_URL}/${firstInterestId}`);
            console.log(`✅ Success: Retrieved interest "${singleInterest.data.data.name}"`);
            console.log('');
        }

        // Test 6: Test category filtering
        console.log('6. Testing category filter GET /api/interests?category=Technology');
        const categoryResults = await axios.get(`${BASE_URL}?category=Technology`);
        console.log(`✅ Success: Found ${categoryResults.data.data.length} technology interests`);
        console.log('');

        // Test 7: Test filtering by active status
        console.log('7. Testing active filter GET /api/interests?isActive=true');
        const activeInterests = await axios.get(`${BASE_URL}?isActive=true`);
        console.log(`✅ Success: Found ${activeInterests.data.data.length} active interests`);
        console.log('');

        console.log('🎉 All tests passed successfully!');
        console.log('\n📋 API Endpoints Summary:');
        console.log('GET /api/interests - Get all interests (with search and filters)');
        console.log('GET /api/interests/categories - Get all categories');
        console.log('GET /api/interests/category/:category - Get interests by category');
        console.log('GET /api/interests/:id - Get single interest by ID');
        console.log('POST /api/interests - Create new interest (Admin only)');
        console.log('PUT /api/interests/:id - Update interest (Admin only)');
        console.log('DELETE /api/interests/:id - Delete interest (Admin only)');

    } catch (error) {
        console.error('❌ Test failed:', error.response?.data || error.message);
        
        if (error.code === 'ECONNREFUSED') {
            console.log('\n💡 Make sure the server is running on http://localhost:5500');
            console.log('   Run: npm start or node server.js');
        }
    }
}

// Run the tests
testInterestsAPI();
