const FreeEducationExam = require('../models/FreeEducationExam');

// Get free education exams for a specific stage
const getFreeEducationExams = async (req, res) => {
    try {
        const { stage } = req.params;
        const { category } = req.query;

        const filter = { stage, isActive: true };
        if (category) filter.category = category;

        const exams = await FreeEducationExam.find(filter).sort({ createdAt: -1 });

        res.status(200).json({
            success: true,
            data: exams
        });
    } catch (error) {
        console.error('Get free education exams error:', error);
        res.status(500).json({
            success: false,
            message: 'Failed to get free education exams',
            error: error.message
        });
    }
};

// Get specific free education exam
const getFreeEducationExam = async (req, res) => {
    try {
        const { examId } = req.params;
        
        const exam = await FreeEducationExam.findById(examId);
        
        if (!exam) {
            return res.status(404).json({
                success: false,
                message: 'Free education exam not found'
            });
        }

        res.status(200).json({
            success: true,
            data: exam
        });
    } catch (error) {
        console.error('Get free education exam error:', error);
        res.status(500).json({
            success: false,
            message: 'Failed to get free education exam',
            error: error.message
        });
    }
};

// Get parent guidance for a specific exam
const getParentGuidance = async (req, res) => {
    try {
        const { examId } = req.params;
        
        const exam = await FreeEducationExam.findById(examId);
        
        if (!exam) {
            return res.status(404).json({
                success: false,
                message: 'Free education exam not found'
            });
        }

        res.status(200).json({
            success: true,
            data: {
                examName: exam.name,
                parentGuidance: exam.parentGuidance,
                studyTips: exam.studyTips,
                importantDates: exam.importantDates
            }
        });
    } catch (error) {
        console.error('Get parent guidance error:', error);
        res.status(500).json({
            success: false,
            message: 'Failed to get parent guidance',
            error: error.message
        });
    }
};

// Create free education exam (Admin only)
const createFreeEducationExam = async (req, res) => {
    try {
        const examData = req.body;
        
        const exam = new FreeEducationExam(examData);
        await exam.save();

        res.status(201).json({
            success: true,
            message: 'Free education exam created successfully',
            data: exam
        });
    } catch (error) {
        console.error('Create free education exam error:', error);
        res.status(500).json({
            success: false,
            message: 'Failed to create free education exam',
            error: error.message
        });
    }
};

// Update free education exam (Admin only)
const updateFreeEducationExam = async (req, res) => {
    try {
        const { examId } = req.params;
        const updateData = req.body;

        const exam = await FreeEducationExam.findByIdAndUpdate(
            examId,
            updateData,
            { new: true, runValidators: true }
        );

        if (!exam) {
            return res.status(404).json({
                success: false,
                message: 'Free education exam not found'
            });
        }

        res.status(200).json({
            success: true,
            message: 'Free education exam updated successfully',
            data: exam
        });
    } catch (error) {
        console.error('Update free education exam error:', error);
        res.status(500).json({
            success: false,
            message: 'Failed to update free education exam',
            error: error.message
        });
    }
};

module.exports = {
    getFreeEducationExams,
    getFreeEducationExam,
    getParentGuidance,
    createFreeEducationExam,
    updateFreeEducationExam
};
