# User Profile API Documentation

## Overview
This documentation covers all user profile-related API endpoints including occupation, personal details, school details, college details, and completion status.

## 🔍 **API Endpoints Summary**

### **1. Occupation Management**

#### **Get Current User's Occupation**
```bash
GET /api/users/occupation
Authorization: Bearer <token>
```

**Response:**
```json
{
    "success": true,
    "message": "Occupation retrieved successfully",
    "data": {
        "userId": "68d26d2c0af08456538bc0f6",
        "name": "John Doe",
        "mobile": "9876543210",
        "occupation": "working professional",
        "hasOccupation": true,
        "occupationCompleted": true,
        "personalDetailsCompleted": false,
        "hasPersonalDetails": false
    }
}
```

#### **Update Occupation**
```bash
POST /api/users/occupation
Authorization: Bearer <token>
Content-Type: application/json

{
    "occupation": "working professional"
}
```

**Valid Occupation Values:**
- `"school student"`
- `"graduate student"`
- `"working professional"`
- `"business"`
- `"public page"`

**Response:**
```json
{
    "success": true,
    "message": "Occupation updated successfully",
    "data": {
        "userId": "68d26d2c0af08456538bc0f6",
        "name": "John Doe",
        "mobile": "9876543210",
        "occupation": "working professional",
        "hasOccupation": true,
        "occupationCompleted": true,
        "personalDetailsCompleted": false,
        "hasPersonalDetails": false
    },
    "debug": {
        "beforeUpdate": {
            "occupation": null,
            "hasOccupation": false,
            "occupationCompleted": false
        },
        "afterUpdate": {
            "occupation": "working professional",
            "hasOccupation": true,
            "occupationCompleted": true
        }
    }
}
```

#### **Get Occupation Status**
```bash
GET /api/users/occupation-status
Authorization: Bearer <token>
```

**Response:**
```json
{
    "success": true,
    "message": "User occupation status retrieved successfully",
    "data": {
        "user": {
            "_id": "68d26d2c0af08456538bc0f6",
            "name": "John Doe",
            "mobile": "9876543210",
            "occupation": "working professional",
            "profession_status": false,
            "occupationCompleted": true,
            "hasOccupation": true
        }
    }
}
```

#### **Delete Occupation**
```bash
DELETE /api/users/occupation
Authorization: Bearer <token>
```

**Response:**
```json
{
    "success": true,
    "message": "Occupation deleted successfully",
    "data": {
        "userId": "68d26d2c0af08456538bc0f6",
        "name": "John Doe",
        "mobile": "9876543210",
        "occupation": null,
        "hasOccupation": false,
        "occupationCompleted": false
    }
}
```

---

### **2. Personal Details & Profile**

#### **Get Current User Profile**
```bash
GET /api/users/profile
Authorization: Bearer <token>
```

**Response:**
```json
{
    "success": true,
    "message": "User profile retrieved successfully",
    "data": {
        "user": {
            "_id": "68d26d2c0af08456538bc0f6",
            "name": "John Doe",
            "mobile": "9876543210",
            "email": "john@example.com",
            "profileImage": "profile_123.jpg",
            "occupation": "working professional",
            "profession_status": false,
            "schoolDetails": {
                "schoolState": "Karnataka",
                "schoolDistrict": "Bangalore",
                "schoolVillage": "Koramangala",
                "schoolCode": "28214900695",
                "schoolName": "ABC School",
                "class10thStatus": "completed",
                "academicYear": "2020"
            },
            "collegeDetails": {
                "graduationStatus": "completed",
                "collegeCode": "S-8897",
                "collegeName": "XYZ College",
                "courseName": "Computer Science",
                "courseSubBranch": "Software Engineering",
                "academicYear": "2023",
                "selectOnYear": "2019"
            },
            "hasPersonalDetails": true,
            "personalDetailsCompleted": true,
            "occupationCompleted": true,
            "hasOccupation": true,
            "createdAt": "2024-01-15T10:00:00.000Z",
            "updatedAt": "2024-01-15T12:00:00.000Z"
        }
    }
}
```

#### **Get User by ID**
```bash
GET /api/users/:userId
Authorization: Bearer <token>
```

**Response:** Same as above but for the specified user ID.

---

### **3. School Details**

#### **Get School Details**
```bash
GET /api/users/school-details/:userId
```

**Response:**
```json
{
    "success": true,
    "message": "School details retrieved successfully",
    "data": {
        "userId": "68d26d2c0af08456538bc0f6",
        "name": "John Doe",
        "mobile": "9876543210",
        "schoolDetails": {
            "schoolState": "Karnataka",
            "schoolDistrict": "Bangalore",
            "schoolVillage": "Koramangala",
            "schoolCode": "28214900695",
            "schoolName": "ABC School",
            "class10thStatus": "completed",
            "academicYear": "2020"
        },
        "occupation": "working professional"
    }
}
```

#### **Update School Details**
```bash
PUT /api/users/school-details/:userId
Authorization: Bearer <token>
Content-Type: application/json

{
    "schoolState": "Karnataka",
    "schoolDistrict": "Bangalore",
    "schoolVillage": "Koramangala",
    "schoolCode": "28214900695",
    "schoolName": "ABC School",
    "class10thStatus": "completed",
    "academicYear": "2020"
}
```

**Response:**
```json
{
    "success": true,
    "message": "School details updated successfully",
    "data": {
        "userId": "68d26d2c0af08456538bc0f6",
        "name": "John Doe",
        "mobile": "9876543210",
        "schoolDetails": {
            "schoolState": "Karnataka",
            "schoolDistrict": "Bangalore",
            "schoolVillage": "Koramangala",
            "schoolCode": "28214900695",
            "schoolName": "ABC School",
            "class10thStatus": "completed",
            "academicYear": "2020"
        },
        "schoolGroupId": "68d26d2c0af08456538bc0f7"
    }
}
```

---

### **4. College Details**

#### **Get College Details**
```bash
GET /api/users/college-details/:userId
```

**Response:**
```json
{
    "success": true,
    "message": "College details retrieved successfully",
    "data": {
        "userId": "68d26d2c0af08456538bc0f6",
        "name": "John Doe",
        "mobile": "9876543210",
        "collegeDetails": {
            "graduationStatus": "completed",
            "collegeCode": "S-8897",
            "collegeName": "XYZ College",
            "courseName": "Computer Science",
            "courseSubBranch": "Software Engineering",
            "academicYear": "2023",
            "selectOnYear": "2019"
        },
        "occupation": "working professional"
    }
}
```

#### **Update College Details**
```bash
PUT /api/users/college-details/:userId
Authorization: Bearer <token>
Content-Type: application/json

{
    "graduationStatus": "completed",
    "collegeCode": "S-8897",
    "collegeName": "XYZ College",
    "courseName": "Computer Science",
    "courseSubBranch": "Software Engineering",
    "academicYear": "2023",
    "selectOnYear": "2019"
}
```

**Response:**
```json
{
    "success": true,
    "message": "College details updated successfully",
    "data": {
        "userId": "68d26d2c0af08456538bc0f6",
        "name": "John Doe",
        "mobile": "9876543210",
        "collegeDetails": {
            "graduationStatus": "completed",
            "collegeCode": "S-8897",
            "collegeName": "XYZ College",
            "courseName": "Computer Science",
            "courseSubBranch": "Software Engineering",
            "academicYear": "2023",
            "selectOnYear": "2019"
        },
        "collegeGroupId": "68d26d2c0af08456538bc0f8"
    }
}
```

---

### **5. User About & Interests**

#### **Get User About**
```bash
GET /api/users/about/:userId
```

#### **Update User About**
```bash
PUT /api/users/about/:userId
Authorization: Bearer <token>
Content-Type: application/json

{
    "about": "I am a software developer passionate about technology and innovation."
}
```

#### **Get User Interests**
```bash
GET /api/users/interests/:userId
```

#### **Update User Interests**
```bash
PUT /api/users/interests/:userId
Authorization: Bearer <token>
Content-Type: application/json

{
    "interests": ["Technology", "Programming", "Music", "Travel"]
}
```

#### **Get Combined About & Interests**
```bash
GET /api/users/about-interests/:userId
```

---

### **6. Completion Status (NEW)**

#### **Get User Completion Status**
```bash
GET /api/users/completion-status/:userId
Authorization: Bearer <token>
```

**Response:**
```json
{
    "success": true,
    "message": "User completion status retrieved successfully",
    "data": {
        "userId": "68d26d2c0af08456538bc0f6",
        "name": "John Doe",
        "mobile": "9876543210",
        "hasOccupation": true,
        "occupation": "working professional",
        "hasPersonalDetails": true,
        "personalDetailsCompleted": true,
        "occupationCompleted": true,
        "profession_status": false,
        "schoolDetails": {
            "schoolState": "Karnataka",
            "schoolDistrict": "Bangalore",
            "schoolVillage": "Koramangala",
            "schoolCode": "28214900695",
            "schoolName": "ABC School",
            "class10thStatus": "completed",
            "academicYear": "2020"
        },
        "collegeDetails": {
            "graduationStatus": "completed",
            "collegeCode": "S-8897",
            "collegeName": "XYZ College",
            "courseName": "Computer Science",
            "courseSubBranch": "Software Engineering",
            "academicYear": "2023",
            "selectOnYear": "2019"
        },
        "about": "I am a software developer passionate about technology and innovation.",
        "interests": ["Technology", "Programming", "Music", "Travel"],
        "completionSummary": {
            "hasOccupation": true,
            "occupation": "working professional",
            "hasPersonalDetails": true,
            "personalDetailsCompleted": true,
            "occupationCompleted": true,
            "hasSchoolDetails": true,
            "hasCollegeDetails": true,
            "hasAbout": true,
            "interestsCount": 4
        }
    }
}
```

---

### **7. Profession Status**

#### **Get Profession Status**
```bash
GET /api/users/profession-status/:userId
```

#### **Update Profession Status**
```bash
PUT /api/users/profession-status/:userId
Authorization: Bearer <token>
Content-Type: application/json

{
    "profession_status": true
}
```

---

## 🔧 **Field Mapping & Flags**

### **Occupation Fields:**
- `occupation`: String (enum: 'school student', 'graduate student', 'working professional', 'business', 'public page')
- `hasOccupation`: Boolean (true if occupation is set)
- `occupationCompleted`: Boolean (true if occupation is completed)

### **Personal Details Fields:**
- `hasPersonalDetails`: Boolean (true if personal details are set)
- `personalDetailsCompleted`: Boolean (true if personal details are completed)
- `profession_status`: Boolean (professional status flag)

### **School Details Object:**
```json
{
    "schoolState": "String",
    "schoolDistrict": "String", 
    "schoolVillage": "String",
    "schoolCode": "String",
    "schoolName": "String",
    "class10thStatus": "String",
    "academicYear": "String"
}
```

### **College Details Object:**
```json
{
    "graduationStatus": "String",
    "collegeCode": "String",
    "collegeName": "String",
    "courseName": "String",
    "courseSubBranch": "String",
    "academicYear": "String",
    "selectOnYear": "String"
}
```

---

## 🐛 **Occupation Update Issue Fix**

### **Problem Identified:**
The occupation update was working correctly, but additional debugging has been added to verify the update process.

### **Fix Applied:**
1. **Enhanced Debugging**: Added comprehensive logging before and after updates
2. **Database Verification**: Added verification by re-fetching the user from database
3. **Debug Response**: Added debug information in the response to track changes
4. **New Completion Status Endpoint**: Created comprehensive endpoint to fetch all fields

### **Testing the Fix:**
```bash
# Test occupation update
curl -X POST "http://135.181.103.182:5500/api/users/occupation" \
  -H "Authorization: Bearer YOUR_TOKEN" \
  -H "Content-Type: application/json" \
  -d '{"occupation": "working professional"}'

# Verify the update
curl -X GET "http://135.181.103.182:5500/api/users/completion-status/USER_ID" \
  -H "Authorization: Bearer YOUR_TOKEN"
```

---

## 📱 **Usage Examples**

### **JavaScript/Node.js**
```javascript
// Update occupation
const updateOccupation = async (token, occupation) => {
    const response = await fetch('http://135.181.103.182:5500/api/users/occupation', {
        method: 'POST',
        headers: {
            'Authorization': `Bearer ${token}`,
            'Content-Type': 'application/json'
        },
        body: JSON.stringify({ occupation })
    });
    return await response.json();
};

// Get completion status
const getCompletionStatus = async (token, userId) => {
    const response = await fetch(`http://135.181.103.182:5500/api/users/completion-status/${userId}`, {
        headers: {
            'Authorization': `Bearer ${token}`
        }
    });
    return await response.json();
};

// Update school details
const updateSchoolDetails = async (token, userId, schoolData) => {
    const response = await fetch(`http://135.181.103.182:5500/api/users/school-details/${userId}`, {
        method: 'PUT',
        headers: {
            'Authorization': `Bearer ${token}`,
            'Content-Type': 'application/json'
        },
        body: JSON.stringify(schoolData)
    });
    return await response.json();
};
```

### **cURL Examples**
```bash
# Get occupation
curl -H "Authorization: Bearer YOUR_TOKEN" \
     "http://135.181.103.182:5500/api/users/occupation"

# Update occupation
curl -X POST \
     -H "Authorization: Bearer YOUR_TOKEN" \
     -H "Content-Type: application/json" \
     -d '{"occupation": "working professional"}' \
     "http://135.181.103.182:5500/api/users/occupation"

# Get completion status
curl -H "Authorization: Bearer YOUR_TOKEN" \
     "http://135.181.103.182:5500/api/users/completion-status/USER_ID"

# Update school details
curl -X PUT \
     -H "Authorization: Bearer YOUR_TOKEN" \
     -H "Content-Type: application/json" \
     -d '{
       "schoolState": "Karnataka",
       "schoolDistrict": "Bangalore",
       "schoolVillage": "Koramangala",
       "schoolCode": "28214900695",
       "schoolName": "ABC School",
       "class10thStatus": "completed",
       "academicYear": "2020"
     }' \
     "http://135.181.103.182:5500/api/users/school-details/USER_ID"
```

---

## ✅ **Features**

- ✅ **Occupation Management**: Create, read, update, delete occupation
- ✅ **Personal Details**: Track completion status and flags
- ✅ **School Details**: Complete school information management
- ✅ **College Details**: Complete college information management
- ✅ **About & Interests**: User bio and interest management
- ✅ **Completion Status**: Comprehensive status tracking
- ✅ **Enhanced Debugging**: Detailed logging for troubleshooting
- ✅ **Database Verification**: Double-check updates in database
- ✅ **Comprehensive API**: All fields accessible through dedicated endpoints
