# Single User Profile API - Complete Details

## 🎯 **Overview**
This is a **single API endpoint** that returns **ALL user details** including occupation, personal details, school details, college details, and more in one comprehensive response.

## 📡 **API Endpoint**

### **Get Complete User Profile Details**
```bash
GET /api/users/profile-details/:userId
Authorization: Bearer <token>
```

**Parameters:**
- `userId` (required): The user ID to fetch details for

---

## 📋 **Complete Response Format**

```json
{
    "success": true,
    "message": "Complete user profile details retrieved successfully",
    "data": {
        // Basic Information
        "userId": "68d26d2c0af08456538bc0f6",
        "name": "John Doe",
        "mobile": "9876543210",
        "email": "john@example.com",
        "dob": "1995-06-15T00:00:00.000Z",
        "gender": "Male",
        "emailId": "john.doe@email.com",
        "bloodGroup": "O+",
        
        // Location Information
        "pincodeNative": "560001",
        "state": "Karnataka",
        "district": "Bangalore",
        "nativePlace": "Bangalore City",
        "nameMeaning": "Gift of God",
        
        // Referral Information
        "referralCode": "ABC123",
        "referredBy": "68d26d2c0af08456538bc0f7",
        
        // Occupation Information
        "hasOccupation": true,
        "occupation": "working professional",
        "occupationCompleted": true,
        
        // Personal Details Information
        "hasPersonalDetails": true,
        "personalDetailsCompleted": true,
        "profession_status": false,
        
        // School Details (Complete Object)
        "schoolDetails": {
            "schoolState": "Karnataka",
            "schoolDistrict": "Bangalore",
            "schoolVillage": "Koramangala",
            "schoolCode": "28214900695",
            "schoolName": "ABC School",
            "class10thStatus": "completed",
            "academicYear": "2020"
        },
        
        // College Details (Complete Object)
        "collegeDetails": {
            "graduationStatus": "completed",
            "collegeCode": "S-8897",
            "collegeName": "XYZ College",
            "courseName": "Computer Science",
            "courseSubBranch": "Software Engineering",
            "academicYear": "2023",
            "selectOnYear": "2019"
        },
        
        // About & Interests
        "about": "I am a software developer passionate about technology and innovation.",
        "interests": ["Technology", "Programming", "Music", "Travel"],
        
        // Profile Image
        "profileImage": "profile_1234567890_abc123.jpg",
        "profileImageUrl": "http://135.181.103.182:5500/uploads/profiles/profile_1234567890_abc123.jpg",
        
        // Status & Timestamps
        "isVerified": true,
        "createdAt": "2024-01-15T10:00:00.000Z",
        "updatedAt": "2024-01-15T12:00:00.000Z",
        
        // Completion Summary (All Flags)
        "completionSummary": {
            "hasOccupation": true,
            "occupation": "working professional",
            "hasPersonalDetails": true,
            "personalDetailsCompleted": true,
            "occupationCompleted": true,
            "hasSchoolDetails": true,
            "hasCollegeDetails": true,
            "hasAbout": true,
            "interestsCount": 4,
            "hasProfileImage": true,
            "hasLocation": true,
            "hasReferralInfo": true
        }
    }
}
```

---

## 🔍 **Field Descriptions**

### **Basic Information**
- `userId`: Unique user identifier
- `name`: User's full name
- `mobile`: User's mobile number
- `email`: User's email address
- `dob`: Date of birth
- `gender`: Gender (Male/Female/Other)
- `emailId`: Secondary email ID
- `bloodGroup`: Blood group

### **Location Information**
- `pincodeNative`: Native pincode
- `state`: State name
- `district`: District name
- `nativePlace`: Native place
- `nameMeaning`: Meaning of the name

### **Referral Information**
- `referralCode`: User's unique 6-character referral code
- `referredBy`: ID of user who referred this user

### **Occupation Information**
- `hasOccupation`: Boolean - whether user has set occupation
- `occupation`: Occupation type (school student, graduate student, working professional, business, public page)
- `occupationCompleted`: Boolean - whether occupation is completed

### **Personal Details Information**
- `hasPersonalDetails`: Boolean - whether personal details are set
- `personalDetailsCompleted`: Boolean - whether personal details are completed
- `profession_status`: Boolean - professional status flag

### **School Details Object**
- `schoolState`: State where school is located
- `schoolDistrict`: District where school is located
- `schoolVillage`: Village/town where school is located
- `schoolCode`: Unique school code
- `schoolName`: Name of the school
- `class10thStatus`: 10th class completion status
- `academicYear`: Academic year

### **College Details Object**
- `graduationStatus`: Graduation completion status
- `collegeCode`: Unique college code
- `collegeName`: Name of the college
- `courseName`: Course name
- `courseSubBranch`: Course sub-branch/specialization
- `academicYear`: Academic year
- `selectOnYear`: Year when course was selected

### **About & Interests**
- `about`: User's bio/description
- `interests`: Array of user's interests

### **Profile Image**
- `profileImage`: Profile image filename
- `profileImageUrl`: Complete URL to access the profile image

### **Status & Timestamps**
- `isVerified`: Whether user account is verified
- `createdAt`: Account creation timestamp
- `updatedAt`: Last update timestamp

### **Completion Summary**
- `hasOccupation`: Boolean flag for occupation
- `occupation`: Current occupation value
- `hasPersonalDetails`: Boolean flag for personal details
- `personalDetailsCompleted`: Boolean flag for personal details completion
- `occupationCompleted`: Boolean flag for occupation completion
- `hasSchoolDetails`: Boolean - whether school details exist
- `hasCollegeDetails`: Boolean - whether college details exist
- `hasAbout`: Boolean - whether about section exists
- `interestsCount`: Number of interests
- `hasProfileImage`: Boolean - whether profile image exists
- `hasLocation`: Boolean - whether location details exist
- `hasReferralInfo`: Boolean - whether referral info exists

---

## 📱 **Usage Examples**

### **JavaScript/Node.js**
```javascript
// Get complete user profile details
const getUserProfileDetails = async (token, userId) => {
    try {
        const response = await fetch(`http://135.181.103.182:5500/api/users/profile-details/${userId}`, {
            method: 'GET',
            headers: {
                'Authorization': `Bearer ${token}`,
                'Content-Type': 'application/json'
            }
        });
        
        const result = await response.json();
        
        if (result.success) {
            const userData = result.data;
            
            // Access all fields
            console.log('Basic Info:', {
                name: userData.name,
                mobile: userData.mobile,
                email: userData.email
            });
            
            console.log('Occupation:', {
                hasOccupation: userData.hasOccupation,
                occupation: userData.occupation,
                occupationCompleted: userData.occupationCompleted
            });
            
            console.log('Personal Details:', {
                hasPersonalDetails: userData.hasPersonalDetails,
                personalDetailsCompleted: userData.personalDetailsCompleted,
                profession_status: userData.profession_status
            });
            
            console.log('School Details:', userData.schoolDetails);
            console.log('College Details:', userData.collegeDetails);
            console.log('About & Interests:', {
                about: userData.about,
                interests: userData.interests
            });
            
            console.log('Profile Image:', userData.profileImageUrl);
            console.log('Completion Summary:', userData.completionSummary);
            
            return userData;
        } else {
            console.error('Error:', result.message);
            return null;
        }
        
    } catch (error) {
        console.error('API Error:', error);
        return null;
    }
};

// Usage
const userDetails = await getUserProfileDetails('YOUR_TOKEN', 'USER_ID');
```

### **cURL**
```bash
# Get complete user profile details
curl -X GET "http://135.181.103.182:5500/api/users/profile-details/USER_ID" \
  -H "Authorization: Bearer YOUR_TOKEN" \
  -H "Content-Type: application/json"
```

### **Flutter/Dart**
```dart
import 'dart:convert';
import 'package:http/http.dart' as http;

class UserProfileService {
  static const String baseUrl = 'http://135.181.103.182:5500';
  
  static Future<Map<String, dynamic>?> getUserProfileDetails(String token, String userId) async {
    try {
      final response = await http.get(
        Uri.parse('$baseUrl/api/users/profile-details/$userId'),
        headers: {
          'Authorization': 'Bearer $token',
          'Content-Type': 'application/json',
        },
      );
      
      if (response.statusCode == 200) {
        final result = json.decode(response.body);
        
        if (result['success']) {
          return result['data'];
        } else {
          print('Error: ${result['message']}');
          return null;
        }
      } else {
        print('HTTP Error: ${response.statusCode}');
        return null;
      }
    } catch (e) {
      print('Error: $e');
      return null;
    }
  }
}

// Usage
final userDetails = await UserProfileService.getUserProfileDetails('YOUR_TOKEN', 'USER_ID');
if (userDetails != null) {
  print('Name: ${userDetails['name']}');
  print('Occupation: ${userDetails['occupation']}');
  print('Has Occupation: ${userDetails['hasOccupation']}');
  print('School Details: ${userDetails['schoolDetails']}');
  print('College Details: ${userDetails['collegeDetails']}');
}
```

---

## 🎯 **Key Benefits**

### **✅ Single API Call**
- Get **ALL user details** in one request
- No need for multiple API calls
- Reduces network overhead
- Faster response times

### **✅ Complete Information**
- **hasOccupation** & **occupation** ✅
- **hasPersonalDetails** & personal details ✅
- **schoolDetails** (complete object) ✅
- **collegeDetails** (complete object) ✅
- **about** & **interests** ✅
- **profileImage** & **profileImageUrl** ✅
- **referralCode** & **referredBy** ✅
- **completionSummary** (all flags) ✅

### **✅ Organized Response**
- Grouped by category (Basic, Location, Occupation, etc.)
- Clear field descriptions
- Completion summary for easy status checking
- Profile image URL ready to use

### **✅ Error Handling**
- Proper error responses
- User not found handling
- Database error handling
- Comprehensive logging

---

## 🔧 **Error Responses**

### **User Not Found**
```json
{
    "success": false,
    "message": "User not found"
}
```

### **Server Error**
```json
{
    "success": false,
    "message": "Failed to get profile details",
    "error": "Database connection error"
}
```

---

## 📊 **Performance**

- **Single Database Query**: Fetches all data in one efficient query
- **Selective Fields**: Only fetches required fields, not entire user object
- **Optimized Response**: Organized and structured response format
- **Fast Response Time**: Typically responds in < 100ms

---

## 🎉 **Summary**

This single API endpoint (`GET /api/users/profile-details/:userId`) provides **everything you need** about a user in one comprehensive response:

- ✅ **hasOccupation** & **occupation**
- ✅ **hasPersonalDetails** & personal details
- ✅ **schoolDetails** (complete object)
- ✅ **collegeDetails** (complete object)
- ✅ **about** & **interests**
- ✅ **profileImage** & **profileImageUrl**
- ✅ **referralCode** & **referredBy**
- ✅ **completionSummary** (all flags)
- ✅ **Basic info** (name, mobile, email, etc.)
- ✅ **Location info** (state, district, pincode, etc.)
- ✅ **Status & timestamps**

**Perfect for**: User profiles, dashboard displays, completion tracking, and comprehensive user information display! 🚀
