const WebSocket = require('ws');

// Test WebSocket authentication
function testWebSocketAuth() {
    console.log('Testing WebSocket Authentication...\n');
    
    // Test 1: Connection without token (should fail)
    console.log('Test 1: Connecting without token...');
    const ws1 = new WebSocket('ws://localhost:5000/ws');
    
    ws1.on('open', () => {
        console.log('❌ FAILED: Connection should have been rejected');
        ws1.close();
    });
    
    ws1.on('close', (code, reason) => {
        if (code === 1008) {
            console.log('✅ PASSED: Connection rejected as expected');
        } else {
            console.log('❌ FAILED: Unexpected close code:', code);
        }
    });
    
    ws1.on('error', (error) => {
        console.log('✅ PASSED: Connection error as expected:', error.message);
    });
    
    // Test 2: Connection with invalid token (should fail)
    setTimeout(() => {
        console.log('\nTest 2: Connecting with invalid token...');
        const ws2 = new WebSocket('ws://localhost:5000/ws?token=invalid-token');
        
        ws2.on('open', () => {
            console.log('❌ FAILED: Connection should have been rejected');
            ws2.close();
        });
        
        ws2.on('close', (code, reason) => {
            if (code === 1008) {
                console.log('✅ PASSED: Invalid token rejected as expected');
            } else {
                console.log('❌ FAILED: Unexpected close code:', code);
            }
        });
        
        ws2.on('error', (error) => {
            console.log('✅ PASSED: Invalid token error as expected:', error.message);
        });
    }, 1000);
    
    // Test 3: Connection with valid token (should succeed)
    setTimeout(() => {
        console.log('\nTest 3: Connecting with valid token...');
        console.log('Note: You need to provide a valid JWT token for this test');
        console.log('To get a valid token, login through the API first');
        
        // This would need a real token from your authentication system
        const validToken = process.env.TEST_JWT_TOKEN || 'your-valid-jwt-token-here';
        
        if (validToken === 'your-valid-jwt-token-here') {
            console.log('⚠️  SKIPPED: No valid token provided. Set TEST_JWT_TOKEN environment variable to test.');
            return;
        }
        
        const ws3 = new WebSocket(`ws://localhost:5000/ws?token=${validToken}`);
        
        ws3.on('open', () => {
            console.log('✅ PASSED: Valid token connection successful');
            
            // Test sending a message
            ws3.send(JSON.stringify({
                type: 'ping'
            }));
        });
        
        ws3.on('message', (data) => {
            const message = JSON.parse(data.toString());
            console.log('✅ PASSED: Received message:', message.type);
            ws3.close();
        });
        
        ws3.on('close', (code, reason) => {
            console.log('Connection closed with code:', code);
        });
        
        ws3.on('error', (error) => {
            console.log('❌ FAILED: Valid token connection error:', error.message);
        });
    }, 2000);
}

// Run the test
testWebSocketAuth();
