const express = require('express');
const router = express.Router();
const {
    getNews,
    getNewsByCategory,
    getNewsBySource,
    searchNews,
    getNewsById,
    getFullArticleContent,
    getCategories,
    getSources,
    refreshNews,
    getTrendingNews,
    getLatestNews,
    getNewsStats,
    clearCache
} = require('../controllers/newsController');

// Get all news with filtering and pagination
// GET /api/news?category=technology&source=newsapi&page=1&limit=20&sortBy=publishedAt&sortOrder=desc
router.get('/', getNews);

// Get news categories (must come before /:id route)
// GET /api/news/categories
router.get('/categories', getCategories);

// Get news sources (must come before /:id route)
// GET /api/news/sources
router.get('/sources', getSources);

// Get trending news (most viewed) (must come before /:id route)
// GET /api/news/trending?limit=10&days=7
router.get('/trending', getTrendingNews);

// Get latest news (must come before /:id route)
// GET /api/news/latest?limit=20
router.get('/latest', getLatestNews);

// Get news statistics (must come before /:id route)
// GET /api/news/stats
router.get('/stats', getNewsStats);

// Search news (must come before /:id route)
// GET /api/news/search?q=artificial intelligence&page=1&limit=20
router.get('/search', searchNews);

// Get news by category
// GET /api/news/category/technology?page=1&limit=20
router.get('/category/:category', getNewsByCategory);

// Get news by source
// GET /api/news/source/newsapi?page=1&limit=20
router.get('/source/:source', getNewsBySource);

// Get full article content (scrape from original URL)
// GET /api/news/:id/full-content
router.get('/:id/full-content', getFullArticleContent);

// Get single news article by ID (must be last)
// GET /api/news/:id
router.get('/:id', getNewsById);

// Refresh news (fetch from APIs)
// POST /api/news/refresh
// Body: { categories: ['technology', 'business'] } (optional)
router.post('/refresh', refreshNews);

// Clear news cache
// POST /api/news/cache/clear
router.post('/cache/clear', clearCache);

module.exports = router;
