const mongoose = require('mongoose');

const examSchema = new mongoose.Schema({
    name: {
        type: String,
        required: true,
        trim: true,
        unique: true
    },
    fullForm: {
        type: String,
        required: true
    },
    description: {
        type: String,
        maxlength: 500
    },
    stage: {
        type: String,
        required: true,
        enum: ['class_6_10', 'class_11_12', 'graduation', 'working_professional']
    },
    category: {
        type: String,
        required: true,
        enum: ['entrance', 'competitive', 'certification', 'international']
    },
    conductingBody: {
        type: String,
        required: true
    },
    difficulty: {
        type: String,
        enum: ['easy', 'medium', 'hard'],
        default: 'medium'
    },
    language: {
        type: [String],
        default: ['English']
    },
    eligibility: {
        minAge: {
            type: Number,
            min: 0,
            max: 100
        },
        maxAge: {
            type: Number,
            min: 0,
            max: 100
        },
        education: {
            type: String,
            required: true
        },
        percentage: {
            type: Number,
            min: 0,
            max: 100
        },
        subjects: {
            type: [String],
            required: function() {
                return this.category === 'entrance' || this.category === 'competitive';
            }
        },
        nationality: {
            type: String,
            default: 'Indian'
        }
    },
    examPattern: {
        mode: {
            type: String,
            enum: ['online', 'offline', 'both'],
            default: 'online'
        },
        duration: {
            type: Number,
            min: 1
        },
        totalMarks: {
            type: Number,
            min: 1
        },
        totalQuestions: Number,
        sections: [{
            name: {
                type: String,
                required: true
            },
            marks: {
                type: Number,
                min: 1
            },
            questions: {
                type: Number,
                min: 1
            },
            duration: {
                type: Number,
                min: 1
            },
            negativeMarking: {
                type: Number,
                default: 0
            }
        }]
    },
    syllabus: [String],
    preparationTime: {
        type: Number,
        min: 1,
        max: 24
    },
    applicationFee: {
        general: {
            type: Number,
            min: 0
        },
        reserved: {
            type: Number,
            min: 0
        },
        currency: {
            type: String,
            default: 'INR'
        }
    },
    importantDates: {
        applicationStart: Date,
        applicationEnd: Date,
        admitCard: Date,
        examDate: Date,
        answerKey: Date,
        resultDate: Date,
        counsellingStart: Date,
        counsellingEnd: Date
    },
    cutoffs: {
        general: {
            type: Number,
            min: 0,
            max: 100
        },
        obc: {
            type: Number,
            min: 0,
            max: 100
        },
        sc: {
            type: Number,
            min: 0,
            max: 100
        },
        st: {
            type: Number,
            min: 0,
            max: 100
        },
        ews: {
            type: Number,
            min: 0,
            max: 100
        }
    },
    examCenters: [String],
    topColleges: [String],
    studyTips: [String],
    previousYearPapers: [String],
    officialWebsite: {
        type: String,
        validate: {
            validator: function(v) {
                return /^https?:\/\/.+/.test(v);
            },
            message: 'Invalid website URL'
        }
    },
    isActive: {
        type: Boolean,
        default: true
    },
    popularity: {
        type: Number,
        default: 0,
        min: 0
    }
}, {
    timestamps: true
});

module.exports = mongoose.model('Exam', examSchema);
