const CareerProgression = require('../models/CareerProgression');

// Get career progression for a specific field
const getCareerProgression = async (req, res) => {
    try {
        const { field } = req.params;
        const { currentLevel } = req.query;

        const filter = { field, isActive: true };
        if (currentLevel) filter.currentLevel = currentLevel;

        const progression = await CareerProgression.find(filter);

        if (progression.length === 0) {
            return res.status(404).json({
                success: false,
                message: 'Career progression not found for this field'
            });
        }

        res.status(200).json({
            success: true,
            data: progression
        });
    } catch (error) {
        console.error('Get career progression error:', error);
        res.status(500).json({
            success: false,
            message: 'Failed to get career progression',
            error: error.message
        });
    }
};

// Get industry switching guidance
const getIndustrySwitching = async (req, res) => {
    try {
        const { fromField, toField } = req.query;

        if (!fromField || !toField) {
            return res.status(400).json({
                success: false,
                message: 'From field and to field are required'
            });
        }

        const switching = await CareerProgression.find({
            'industrySwitching.fromField': fromField,
            'industrySwitching.toField': toField,
            isActive: true
        });

        if (switching.length === 0) {
            return res.status(404).json({
                success: false,
                message: 'Industry switching guidance not found'
            });
        }

        res.status(200).json({
            success: true,
            data: switching
        });
    } catch (error) {
        console.error('Get industry switching error:', error);
        res.status(500).json({
            success: false,
            message: 'Failed to get industry switching guidance',
            error: error.message
        });
    }
};

// Get certification impact
const getCertificationImpact = async (req, res) => {
    try {
        const { field } = req.params;

        const progression = await CareerProgression.findOne({ 
            field, 
            isActive: true 
        }).select('certifications');

        if (!progression) {
            return res.status(404).json({
                success: false,
                message: 'Certification data not found for this field'
            });
        }

        res.status(200).json({
            success: true,
            data: progression.certifications
        });
    } catch (error) {
        console.error('Get certification impact error:', error);
        res.status(500).json({
            success: false,
            message: 'Failed to get certification impact',
            error: error.message
        });
    }
};

// Create career progression (Admin only)
const createCareerProgression = async (req, res) => {
    try {
        const progressionData = req.body;
        
        const progression = new CareerProgression(progressionData);
        await progression.save();

        res.status(201).json({
            success: true,
            message: 'Career progression created successfully',
            data: progression
        });
    } catch (error) {
        console.error('Create career progression error:', error);
        res.status(500).json({
            success: false,
            message: 'Failed to create career progression',
            error: error.message
        });
    }
};

// Update career progression (Admin only)
const updateCareerProgression = async (req, res) => {
    try {
        const { progressionId } = req.params;
        const updateData = req.body;

        const progression = await CareerProgression.findByIdAndUpdate(
            progressionId,
            updateData,
            { new: true, runValidators: true }
        );

        if (!progression) {
            return res.status(404).json({
                success: false,
                message: 'Career progression not found'
            });
        }

        res.status(200).json({
            success: true,
            message: 'Career progression updated successfully',
            data: progression
        });
    } catch (error) {
        console.error('Update career progression error:', error);
        res.status(500).json({
            success: false,
            message: 'Failed to update career progression',
            error: error.message
        });
    }
};

module.exports = {
    getCareerProgression,
    getIndustrySwitching,
    getCertificationImpact,
    createCareerProgression,
    updateCareerProgression
};
