# School Details Model Refactoring

## Overview
This refactoring separates school details into two distinct models based on user type:

1. **SchoolDetails** - For current students (with `currentClass` field)
2. **SchoolDetailsWithoutCurrentClass** - For graduates and working professionals (without `currentClass` field)

## Models

### 1. SchoolDetails (Existing - Updated)
- **Purpose**: For current students (6th to 10th grade)
- **Key Field**: `currentClass` (required)
- **Usage**: Used by current students only
- **Location**: `backend/models/SchoolDetails.js`

### 2. SchoolDetailsWithoutCurrentClass (New)
- **Purpose**: For graduates and working professionals
- **Key Difference**: No `currentClass` field
- **Usage**: Referenced by GraduateDetails and WorkingProfessionalDetails models
- **Location**: `backend/models/SchoolDetailsWithoutCurrentClass.js`

## Updated Models

### GraduateDetails
- **Before**: Embedded school details fields
- **After**: References `SchoolDetailsWithoutCurrentClass` via `schoolDetailsId`
- **Benefits**: 
  - Eliminates data duplication
  - Maintains referential integrity
  - Easier to maintain and update

### WorkingProfessionalDetails
- **Before**: Embedded school details fields
- **After**: References `SchoolDetailsWithoutCurrentClass` via `schoolDetailsId`
- **Benefits**: 
  - Eliminates data duplication
  - Maintains referential integrity
  - Easier to maintain and update

## API Endpoints

### School Details (Current Students)
- **Base URL**: `/api/school-details`
- **Controller**: `schoolDetailsController.js`
- **Model**: `SchoolDetails`

### School Details Without Current Class (Graduates/Working Professionals)
- **Base URL**: `/api/school-details-without-current-class`
- **Controller**: `schoolDetailsWithoutCurrentClassController.js`
- **Model**: `SchoolDetailsWithoutCurrentClass`

## Usage Examples

### For Current Students
```javascript
// Create/Update school details for current student
POST /api/school-details
{
  "currentClass": "9th",
  "schoolPincode": "110001",
  "schoolState": "Delhi",
  "schoolDistrict": "Central Delhi",
  "schoolVillage": "Connaught Place",
  "schoolName": "Delhi Public School",
  "boardType": "CBSE",
  "schoolType": "Co-ed",
  "yearOfJoiningSchool": 2020,
  "section": "A"
}
```

### For Graduates/Working Professionals
```javascript
// Step 1: Create/Update school details
POST /api/school-details-without-current-class
{
  "schoolPincode": "110001",
  "schoolState": "Delhi",
  "schoolDistrict": "Central Delhi",
  "schoolVillage": "Connaught Place",
  "schoolName": "Delhi Public School",
  "boardType": "CBSE",
  "schoolType": "Co-ed",
  "yearOfJoiningSchool": 2018,
  "section": "A"
}

// Step 2: Create/Update graduate details (references school details)
POST /api/graduate-details
{
  "schoolDetails": {
    "schoolPincode": "110001",
    "schoolState": "Delhi",
    // ... other school fields
  },
  "collegePincode": "110001",
  "collegeState": "Delhi",
  // ... other college fields
}
```

## Database Relationships

```
User
├── SchoolDetails (current students only)
│   └── currentClass (required)
│
├── SchoolDetailsWithoutCurrentClass (graduates/working professionals)
│   └── No currentClass field
│
├── GraduateDetails
│   └── schoolDetailsId → SchoolDetailsWithoutCurrentClass
│
└── WorkingProfessionalDetails
    └── schoolDetailsId → SchoolDetailsWithoutCurrentClass
```

## Benefits of This Refactoring

1. **Data Integrity**: Eliminates duplication of school details across models
2. **Maintainability**: Single source of truth for school details
3. **Flexibility**: Easy to add new fields to school details without updating multiple models
4. **Performance**: Better indexing and query optimization
5. **Consistency**: Standardized school details structure across all user types

## Migration Notes

- Existing data will need to be migrated to the new structure
- Frontend applications will need to be updated to use the appropriate endpoints
- The `currentClass` field is now only required for current students
- Graduates and working professionals use the separate school details model

## Controllers

### schoolDetailsController.js
- Handles CRUD operations for current students' school details
- Includes `currentClass` validation

### schoolDetailsWithoutCurrentClassController.js
- Handles CRUD operations for graduates/working professionals' school details
- No `currentClass` field handling

### graduateDetailsController.js (Updated)
- Now creates/updates school details separately
- References school details via `schoolDetailsId`
- Populates school details in GET operations

### workingProfessionalController.js (Updated)
- Now creates/updates school details separately
- References school details via `schoolDetailsId`
- Populates school details in GET operations

## Routes

### School Details Routes
- `/api/school-details` - Current students only

### School Details Without Current Class Routes
- `/api/school-details-without-current-class` - Graduates/working professionals

Both route sets include:
- POST - Create/Update
- GET - Retrieve
- PUT - Update
- DELETE - Delete
- GET /completion - Get completion status
- GET /schools - Get schools by pincode
- GET /all - Admin: Get all records
- GET /:id - Admin: Get by ID
